from fastapi import FastAPI, Depends, UploadFile, File, Form
from sqlalchemy.orm import Session
from fastapi_mail import FastMail, MessageSchema, ConnectionConfig
from .database import Base, engine, SessionLocal
from . import models, schemas, crud, auth
import shutil, os

app = FastAPI()
Base.metadata.create_all(bind=engine)

UPLOAD_DIR = "uploads"
os.makedirs(os.path.join(UPLOAD_DIR, "adhar_front"), exist_ok=True)
os.makedirs(os.path.join(UPLOAD_DIR, "adhar_back"), exist_ok=True)
os.makedirs(os.path.join(UPLOAD_DIR, "marksheet"), exist_ok=True)

conf = ConnectionConfig(
    MAIL_USERNAME="codeinlastbench@gmail.com",
    MAIL_PASSWORD="tgdk ufio phln qdvo",
    MAIL_FROM="codeinlastbench@gmail.com",
    MAIL_PORT=587,
    MAIL_SERVER="smtp.gmail.com",
    MAIL_STARTTLS=True,
    MAIL_SSL_TLS=False,
    USE_CREDENTIALS=True
)


def get_db():
    db = SessionLocal()
    try:
        yield db
    finally:
        db.close()

@app.post("/signup")
def signup(user: schemas.UserCreate, db: Session = Depends(get_db)):
    return crud.create_user(db, user)

@app.post("/login")
def login(user: schemas.UserLogin, db: Session = Depends(get_db)):
    db_user = crud.authenticate_user(db, user.email, user.password)
    if not db_user:
        return {"error": "Invalid credentials"}
    token = auth.create_token({"sub": db_user.email})
    return {"access_token": token, "token_type": "bearer"}

@app.post("/register")
async def register(
    cname: str = Form(...),
    husband_name: str = Form(...),
    address: str = Form(...),
    education: str = Form(...),
    dob: str = Form(...),
    mobile: str = Form(...),
    email: str = Form(...),
    adhar: str = Form(...),
    adhar_front: UploadFile = File(...),
    adhar_back: UploadFile = File(...),
    marksheet: UploadFile = File(...),
    db: Session = Depends(get_db)
):
    def save_file(file: UploadFile, subfolder: str):
        path = os.path.join(UPLOAD_DIR, subfolder, file.filename)
        with open(path, "wb") as f:
            shutil.copyfileobj(file.file, f)
        return path

    files = {
        "adhar_front": save_file(adhar_front, "adhar_front"),
        "adhar_back": save_file(adhar_back, "adhar_back"),
        "marksheet": save_file(marksheet, "marksheet"),
    }

    reg = schemas.RegistrationBase(
        cname=cname,
        husband_name=husband_name,
        address=address,
        education=education,
        dob=dob,
        mobile=mobile,
        email=email,
        adhar=adhar
    )
    return crud.save_registration(db, reg, files)

@app.get("/registrations")
def get_registrations(db: Session = Depends(get_db)):
    return crud.get_all_registrations(db)


@app.post("/subscribe")
def subscribe(sub: schemas.SubscriberCreate, db: Session = Depends(get_db)):
    return crud.add_subscriber(db, sub.email)

@app.post("/notify")
def notify_all(notification: schemas.NotificationCreate, db: Session = Depends(get_db)):
    notif = crud.create_notification(db, notification)
    
    subscribers = crud.get_all_subscribers(db)  # ✅ only existing subscribers
    emails = [s.email for s in subscribers]

    if emails:
        message = MessageSchema(
            subject=notification.title,
            recipients=emails,
            body=notification.message,
            subtype="plain"
        )
        fm = FastMail(conf)
        fm.send_message(message)

    return {"message": "Notification sent", "title": notification.title}


@app.post("/contact")
def submit_contact(contact: schemas.ContactCreate, db: Session = Depends(get_db)):
    return crud.create_contact(db, contact)

@app.get("/notifications")
def get_notifications(db: Session = Depends(get_db)):
    return crud.get_all_notifications(db)

@app.get("/contacts")
def get_contacts(db: Session = Depends(get_db)):
    return crud.get_all_contacts(db)
